#!/usr/bin/sh

# 
# File: setup.sh
# Version: 0.1
# Authors: Stéphane Cousot <http://www.ubaa.net/>
# License: setup.sh is released under GNU General Public License, version 3 or later
#
# ----------------------------------------------------------------------------------
#
# This file is part of the piview project.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#



if [ `whoami` != 'root' ]; then # && [ `whoami` != 'http' ]; then
	echo "Permission denied"
	echo "you must be root to run this script."
	exit 1
fi


DEFAULT_HOSTANME="piview"
ETHERNET_PROFILE="/etc/systemd/network/eth0.network"
ETHTETHERING_PROFILE="/etc/systemd/network/eth1.network"
USBTETHERING_PROFILE="/etc/systemd/network/usb0.network"



# load (copy first) configuration file
if [ ! -f /boot/live-stream ]; then
	cp /srv/http/conf/script /boot/live-stream
	chmod +x /boot/live-stream
fi

# load (copy first) configuration file
if [ ! -f /boot/settings.txt ]; then
	echo "Copy default configuration"
	cp /srv/http/conf/default /boot/settings.txt
fi
source /boot/settings.txt




# ---------------------------------------------------------------------------------------------------------------------
#	HOSTNAME
# ---------------------------------------------------------------------------------------------------------------------

# change hostname
sethostname()
{
	if [ ! "$HOSTNAME" ]; then
		echo "Invalid hostname, set to the defaut '$DEFAULT_HOSTANME'"
		HOSTNAME=$DEFAULT_HOSTANME
	fi
	if [ `hostname` != `echo ${HOSTNAME,,}` ]; then
		echo "Set hostname to '$HOSTNAME'"
		hostnamectl set-hostname ${HOSTNAME,,}
		echo "Reboot" 
		sudo systemctl reboot -i
	else
		echo "Hostname already set to '$HOSTNAME'" 
	fi
}




# ---------------------------------------------------------------------------------------------------------------------
#	ETHERNET & USB TETHERING
#	https://wiki.archlinux.org/index.php/Systemd-networkd
#	require systemd-networkd service: systemctl enable systemd-networkd.service
# ---------------------------------------------------------------------------------------------------------------------


# write ethernet profile
eth_profile()
{
	
	DHCP=`grep -s DHCP $ETHERNET_PROFILE`
	ADDR=`grep -E -so "^Address=.*" $ETHERNET_PROFILE`
	IP=`echo  $ADDR | grep -E -so "([0-9]{1,3}[\.]){3}[0-9]{1,3}"`
	

	# dynamic configuration
	if [[ "$STATIC_ADDRESS" == "" &&  "$DHCP" != "DHCP=yes" ]]; then
		echo "Write dynamic ethernet profile"
		cat << EOF > $ETHERNET_PROFILE
[Match]
Name=eth0

[Network]
DHCP=yes
EOF
	fi

	# static configuration
	if [[ "$STATIC_ADDRESS" != "" &&  "$IP" != "$STATIC_ADDRESS" ]]; then
		echo "Write static ethernet profile"
		cat << EOF > $ETHERNET_PROFILE
[Match]
Name=eth0

[Network]
DHCP=no
Address=$STATIC_ADDRESS/24
EOF
	fi

}

# write USB tethering profiles (for IOS and Androïd)
# https://wiki.archlinux.org/index.php/IPhone_Tethering
usb_profile()
{
	# IOS
	if [ ! `grep -s DHCP $ETHTETHERING_PROFILE` ]; then
		echo "Write dynamic ethernet tethering profile"
		cat << EOF > $ETHTETHERING_PROFILE
[Match]
Name=eth1

[Network]
DHCP=both
EOF
	fi

	# Androïd
	if [ ! `grep -s DHCP $USBTETHERING_PROFILE` ]; then
		echo "Write dynamic USB tethering profile"
		cat << EOF > $USBTETHERING_PROFILE
[Match]
Name=usb0

[Network]
DHCP=both
EOF
	fi

}




# ---------------------------------------------------------------------------------------------------------------------
#	REMOTE ACCESS
# ---------------------------------------------------------------------------------------------------------------------

HTTP_PORT=""
SSH_PORT=""

doremote()
{
	case $@ in
		start )
			remote_start
			;;
		stop )
			remote_stop
			;;
		scan )
			remote_scan
			;;
		check )
			remote_check
			;;
		* )
			usage
			exit 1
			;;
	esac
}

remote_scan()
{
	HTTP_PORT=""
	SSH_PORT=""

	# list ports
	RULES=`upnpc -l`
	HTTP_PORT=`echo "$RULES" | pcregrep -o1 -e "TCP +?([0-9]+).*piview web interface"`
	SSH_PORT=`echo "$RULES"  | pcregrep -o1 -e "TCP +?([0-9]+).*piview SSH"`

	if [ "$HTTP_PORT" ]; then
		echo "`hostname` HTTP port redirection: $HTTP_PORT"
		echo "`hostname` SSH  port redirection: $SSH_PORT"
	fi
}

remote_stop()
{

	# retrieve previous ports from UPnP config
	remote_scan > /dev/null
	# list local redirection
	RULES=`iptables -t nat -n -L`


	if [ "$HTTP_PORT" ]; then
		echo -n "delete port redirection $HTTP_PORT ... "
		upnpc -d $HTTP_PORT TCP > /dev/null
		HTTP_RULE=`echo "$RULES" | grep "tcp dpt:$HTTP_PORT"`
		if [ "$HTTP_RULE" ]; then iptables -t nat -D PREROUTING -p tcp --dport $HTTP_PORT -j REDIRECT --to-port 80; fi
		echo "done"
	fi
	if [ "$SSH_PORT" ]; then
		echo -n "delete port redirection $SSH_PORT ... "
		upnpc -d $SSH_PORT TCP > /dev/null
		SSH_RULE=`echo "$RULES" | grep "tcp dpt:$SSH_PORT"`
		if [ "$SSH_RULE" ]; then iptables -t nat -D PREROUTING -p tcp --dport $SSH_PORT -j REDIRECT --to-port 22; fi
		echo "done"
	fi
}

remote_start()
{

	# remove previous configuration first
	remote_stop

	# define ports number : ramdomly, in range of 49152 to 65535
	# or using user settings
	if [ ! "$REMOTE_ACCESS_HTTP_PORT" ]; then HTTP_PORT=`echo $[ 49152 + $[ RANDOM % 16383 ]]`;
	else HTTP_PORT=$REMOTE_ACCESS_HTTP_PORT; fi
	if [ ! "$REMOTE_ACCESS_SSH_PORT" ]; then SSH_PORT=`echo $[ 49152 + $[ RANDOM % 16383 ]]`;
	else SSH_PORT=$REMOTE_ACCESS_SSH_PORT; fi

	# add upnp rules on router
	upnpc -e "piview web interface (80)" -r $HTTP_PORT TCP > /dev/null
	upnpc -e "piview SSH (21)" -r $SSH_PORT TCP > /dev/null

	# check / ouput result
	remote_scan

	# add local filters
	if [ "$HTTP_PORT" ]; then
		iptables -t nat -A PREROUTING -p tcp --dport $HTTP_PORT -j REDIRECT --to-port 80
	fi
	if [ "$SSH_PORT" ]; then
		iptables -t nat -A PREROUTING -p tcp --dport $SSH_PORT -j REDIRECT --to-port 22
	fi

}

# check and refresh remote access configuration
remote_check()
{

	if [ "$REMOTE_ACCESS" == "yes" ]; then
		
		# check UPnP configuration
		remote_scan > /dev/null

		if [ -z "$HTTP_PORT" ]; then RESTART=1; fi
		if [[ "$REMOTE_ACCESS_HTTP_PORT" && "$REMOTE_ACCESS_HTTP_PORT" != "$HTTP_PORT" ]]; then RESTART=1; fi
		if [[ "$REMOTE_ACCESS_SSH_PORT"  && "$REMOTE_ACCESS_SSH_PORT"  != "$SSH_PORT"  ]]; then RESTART=1; fi

		# (re)configure UPnP
		if [ "$RESTART" ]; then
			echo "add router UPnP rules"
			remote_start
			exit 0
		fi


		# check local filters
		RULES=`iptables -t nat -n -L`
		if [ "$HTTP_PORT" ]; then
			HTTP_RULE=`echo "$RULES" | grep "tcp dpt:$HTTP_PORT"`
			if [ -z "$HTTP_RULE" ]; then
				echo "add local HTTP redirection to : $HTTP_PORT"
				iptables -t nat -A PREROUTING -p tcp --dport $HTTP_PORT -j REDIRECT --to-port 80
			fi
		fi
		if [ "$SSH_PORT" ]; then
			SSH_RULE=`echo "$RULES" | grep "tcp dpt:$SSH_PORT"`
			if [ -z "$SSH_RULE" ]; then
				echo "add local SSH redirection to  : $SSH_PORT"
				iptables -t nat -A PREROUTING -p tcp --dport $SSH_PORT -j REDIRECT --to-port 22
			fi
		fi
	fi

}




# ---------------------------------------------------------------------------------------------------------------------
#	MAIN FUNCTIONS
# ---------------------------------------------------------------------------------------------------------------------

usage()
{
	echo "$0 {COMMAND} ..."
	echo ""
	echo "piview configuration manager"
	echo 
	echo "  -h --help                       show this help"
	echo "  --update                        all configuration"
	echo "  --hostname                      set hostname and reboot"
	echo "  --profile usb|eth               configure network link profile"
	echo "  --remote start|stop|scan|check  add or delete ports redirection"
	echo "                                  i.e. enable/disable remote access"
}


update()
{
	# update network profiles
	eth_profile
	usb_profile
	systemctl restart systemd-networkd
}

configure()
{
	case $@ in
		eth )
			eth_profile
			;;
		usb )
			usb_profile
			;;
		* )
			usage
			exit 1
			;;
	esac
}

case $1 in
	--update )
		update
		;;
	--hostname )
		sethostname
		;;
	--profile )
		configure $2
		;;
	--remote )
		doremote $2
		;;
	-h|* )
		usage
		exit 1
		;;
esac
exit $?


